import os
import re

def contains_chinese(text):
    """检查字符串是否包含中文字符"""
    return any('\u4e00' <= ch <= '\u9fff' for ch in text)

def extract_translation_after_marker(lines, marker="【英语翻译】"):
    """
    从文本行中找到指定标记，并返回其后的第一行非空英文内容。
    """
    found_marker = False
    for i, line in enumerate(lines):
        if marker in line:
            found_marker = True
            # 从标记的下一行开始查找
            for j in range(i + 1, len(lines)):
                candidate = lines[j].strip()
                # 如果找到非空行且包含英文字符，则返回
                if candidate and re.search('[a-zA-Z]', candidate):
                    return candidate
            break # 标记找到了，但后面没有合适的英文行
    return None

def sanitize_filename(text):
    """
    清理字符串，移除文件名中不允许的字符。
    """
    # 移除 Windows 文件名中不允许的字符: \ / : * ? " < > |
    # 同时移除一些可能由OCR或复制粘贴带来的特殊空白字符
    cleaned_text = re.sub(r'[\\/:*?"<>|]', '', text)
    cleaned_text = re.sub(r'[\u200b\u200c\u200d\u200e\u200f\ufeff]', '', cleaned_text) # 移除零宽字符
    return cleaned_text.strip() # 最后再去除首尾空格

def replace_chinese_in_filename(original_filename, replacement_text):
    """
    替换文件名中的所有中文字符为指定的英文文本，并保留其他部分。
    """
    name, ext = os.path.splitext(original_filename)
    # 使用替换文本替换文件名中的所有中文字符
    new_name_without_chinese = re.sub(r'[\u4e00-\u9fff]+', replacement_text, name)
    return new_name_without_chinese + ext

def rename_txt_files_recursively(root_folder_path):
    """
    递归遍历指定根目录及其所有子目录，重命名符合条件的 .txt 文件。
    """
    processed_count = 0
    skipped_count = 0
    error_count = 0

    print(f"开始递归处理目录: {root_folder_path}")
    
    try:
        # os.walk 遍历目录树，dirpath是当前目录路径，dirnames是子目录列表，filenames是文件列表
        for dirpath, dirnames, filenames in os.walk(root_folder_path):
            print(f"\n--- 正在扫描目录: {dirpath} ---") # 显示当前正在处理的目录

            for filename in filenames:
                # 检查文件是否是 .txt 结尾且文件名中包含中文字符
                if filename.lower().endswith('.txt') and contains_chinese(filename):
                    full_path = os.path.join(dirpath, filename)
                    print(f"  正在处理文件: {filename}")
                    try:
                        # 读取文件前50行
                        with open(full_path, 'r', encoding='utf-8') as f:
                            first_50_lines = [f.readline() for _ in range(50)]
                    except UnicodeDecodeError:
                        print(f"     ⚠️ 编码错误，跳过文件: {filename} (请确保文件是UTF-8编码)")
                        skipped_count += 1
                        continue
                    except Exception as e:
                        print(f"     ❌ 无法读取 {filename}: {e}")
                        error_count += 1
                        continue

                    english_line = extract_translation_after_marker(first_50_lines)
                    if not english_line:
                        print(f"     ℹ️ {filename} 未找到【英语翻译】后的第一行英文内容，跳过。")
                        skipped_count += 1
                        continue

                    english_sanitized = sanitize_filename(english_line) # 传入未strip的，让sanitize_filename处理
                    
                    # 检查替换后的英文是否为空，避免生成空文件名
                    if not english_sanitized:
                        print(f"     ℹ️ 提取的英文内容为空或无效，跳过文件: {filename}")
                        skipped_count += 1
                        continue

                    new_filename = replace_chinese_in_filename(filename, english_sanitized)
                    new_path = os.path.join(dirpath, new_filename) # 注意这里使用 dirpath 来构建新路径

                    # 避免重命名为相同的文件名（如果中文部分被替换后和原文件名相同）
                    if new_path == full_path:
                        print(f"     ℹ️ 文件名替换后没有变化，跳过: {filename}")
                        skipped_count += 1
                        continue

                    # 检查目标文件是否已存在，避免覆盖
                    if os.path.exists(new_path):
                        print(f"     ⚠️ 目标文件 '{new_filename}' 已存在，跳过重命名以避免覆盖: {filename}")
                        skipped_count += 1
                        continue

                    try:
                        os.rename(full_path, new_path)
                        print(f"     ✅ 成功重命名: '{filename}' -> '{new_filename}'")
                        processed_count += 1
                    except Exception as e:
                        print(f"     ❌ 重命名失败 {filename} -> {new_filename}: {e}")
                        error_count += 1
                # else: # 如果不需要打印跳过非txt或不含中文的文件，可以注释掉这部分
                #     print(f"  跳过文件 (非txt或不含中文): {filename}")
                #     skipped_count += 1
        
    except FileNotFoundError:
        print(f"❌ 错误: 根目录 '{root_folder_path}' 不存在。")
        return
    except PermissionError:
        print(f"❌ 错误: 没有权限访问根目录 '{root_folder_path}'。")
        return
    except Exception as e:
        print(f"❌ 错误: 遍历目录时发生未知错误: {e}")
        return

    print("\n--- 处理完成 ---")
    print(f"成功重命名文件数量: {processed_count}")
    print(f"跳过文件数量: {skipped_count}")
    print(f"处理失败文件数量: {error_count}")
    print("------------------")


if __name__ == "__main__":
    # 使用 strip('"').strip("'") 移除用户可能输入的引号
    folder = input("Please enter the default directory path (e.g., D:\MyTexts or /Users/Me/Documents)）：").strip('"').strip("'")
    
    if os.path.isdir(folder):
        rename_txt_files_recursively(folder) # 调用新的递归函数
    else:
        print(f"❌ 目录无效: '{folder}'。请检查路径是否正确。")
    
    # 这行代码会使窗口在程序执行完毕后保持打开，直到用户按下回车键
    input("\n按回车键退出程序...")
